#!/bin/bash

set -eux

# Pixel format for raw image. Consider rgb, bgr, rgba, uyvy
imgfmt=rgb
# Raw image bit depth.
bitdepth=16
# Chroma subsampling, used only for uyvy
sampling=4:4:4
# Raw audio format - that is, how we interpret the raw pixels as audio.
# Consider u8, s8, alaw. Wider formats like u16be, s16le are inherently
# very noisy.
afmt="s8"
# Audio codec. Consider vorbis, opus, speex, mp3
codec=speex
# Codec bitrate. Affects fidelity of output image. ~8k-200k or so
bitrate=32k
# Audio filters for FFMPEG.
# Comma seperated; arguments : seperated
# For instance, delay is used to pan the image over for codecs where there is
# some delay before the codec starts giving output
audiofilters="adelay=delays=1500S:all=1"
# Final output format, doesn't affect image processing at all
fmt="webp"

# Select directory into which to do output
outdir="$(dirname "/tmp/$1")"
# Compose a semi-unique output name
outputname="$1.$imgfmt.$bitdepth.$sampling.$afmt.$codec.$bitrate.$fmt"
mkdir -p "$outdir"
rm -f "/tmp/$outputname" "$outdir/"tmp.*

# Need the size since we'll be using raw image files with no headers
size="$(identify -format "%[fx:w]x%[fx:h]" "$1")"

# Make a raw image
convert "$1" -size "$size" -depth $bitdepth -sampling-factor $sampling \
	$imgfmt:"$outdir/tmp.1"

# Count bytes in the raw image
bytesize="$(du --bytes "$outdir/tmp.1" | cut -f1)"

# Encode with the selected codec, appling selected filters
ffmpeg -f "$afmt" -ar 44100 -ac 2 -i "$outdir/tmp.1" -f matroska -strict -2 \
	-af "$audiofilters" \
	-acodec "$codec" -ab "$bitrate" "$outdir/tmp.2"

# Decode to raw audio, twice. We do this because sometimes the decoding process
# loses a few bytes, which would cause image conversion to fail.
ffmpeg -f matroska -stream_loop 2 -i "$outdir/tmp.2" -f "$afmt" -ar 44100 -ac 2 "$outdir/tmp.3"

# Cut down the oversized output to the correct size, captured above
head --bytes $bytesize "$outdir/tmp.3" > "$outdir/tmp.4"

# Convert to final desired image format
convert -size "$size" -depth $bitdepth -sampling-factor $sampling $imgfmt:"$outdir/tmp.4" "/tmp/$outputname"

# Output image in terminal
timg "/tmp/$outputname"
echo "/tmp/$outputname"
